/* AddPath - Add directory to PATH-like env variable
   Default is PATH

   Copyright (c) 2005-2024 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License.  The GPL Software License can be found in
   gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2005-05-31 SHL Baseline
   2013-03-18 SHL Avoid error if no match and no trailing slash
   2014-06-12 SHL Sync with templates
   2016-07-18 SHL Sync with templates
   2016-12-15 SHL Convert to Globals style
   2016-12-15 SHL Match slashes
   2017-11-29 SHL Sync with templates
   2017-11-29 SHL Correct addpath -s loop if run twice
   2019-06-29 SHL Sync with templates
   2021-12-22 SHL Support -c create if new
   2024-11-18 SHL Sync with templates
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.1 2024-11-18'

Globals = 'gAddWhere gAllowedEnvVarChars gCmdName gCreateIfNew',
	  'gDirectory gDbgLvl gEnv gEnvVar gForce gVerbose gVersion'

call Initialize

gAllowedEnvVarChars = xrange('A','Z') || xrange('a','z') || xrange('0','9') || '_'

Main:

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  u = translate(gEnvVar)
  select
  when u == 'LIBPATH' | u == 'BEGINLIBPATH' then do
    curPath = SysQueryExtLibPath('B')
    what = 'blp'
  end
  when u == 'ENDLIBPATH' then do
    curPath = SysQueryExtLibPath('E')
    what = 'elp'
  end
  otherwise
    curPath = value(gEnvVar,, gEnv)
    what = 'env'
  end

  dir4Cmp = translate(translate(gDirectory), '\', '/')	/* UC and DOSify */

  /* Check directory already in path list */
  i = 1
  do forever
    curPath4Cmp = translate(translate(curPath), '\', '/')	/* UC and DOSify */
    i = pos(dir4Cmp, curPath4Cmp, i)
    if i = 0 then
      leave				/* Not in path */
    l = length(dir4Cmp)
    if i + l - 1 = length(curPath) then
      sep = 0				/* Got match with no trailing ; */
    else do
      if substr(curPath, i + l, 1) == ';' then
	sep = 1
      else do
	/* False positive */
	i = i + l			/* Point after match */
	i = pos(curPath, ';', i)	/* Find next ; */
	if i = 0 then
	  leave				/* Done */
	i = i + 1			/* Point after ; */
	iterate				/* Try again */
      end
    end
    /* Got match */
    if \ gForce then
      leave
    /* Delete match and trailing ; if present */
    curPath = substr(curPath, 1, i - 1) || substr(curPath, i + l + sep)
  end /* forever */

  if i \= 0 then
    say '*' gDirectory 'already in' gEnvVar
  else do
    /* Match slashes */
    if pos('/', curPath) > 0 then
      gDirectory = translate(gDirectory, '/', '\')	/* Assume want forward slashes */
    else if pos('\', curPath) > 0 then
      gDirectory = translate(gDirectory, '\', '/')	/* Assume want backslashes */

    if gAddWhere \= 's' then do
      /* Prefix is default */
      curPath = gDirectory || ';' ||curPath
      how = 'prepended'
    end
    else do
      how = 'appended'
      if right(curPath, 1) == ';' then
	curPath = curPath || gDirectory || ';'
      else if curPath == '' then
	curPath = gDirectory
      else
	curPath = curPath || ';' || gDirectory
    end

    /* Update */
    select
    when what == 'blp' then
      call SysSetExtLibPath curPath, 'B'
    when what == 'elp' then
      call SysSetExtLibPath curPath, 'E'
    otherwise
      call value gEnvVar, curPath, gEnv
    end

    if gVerbose then
      say '*' gDirectory how 'to' gEnvVar
    if gDbgLvl > 0 then
      'set' gEnvVar
  end

  exit

/* end main */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted
  /* Preset defaults */
  gCreateIfNew = 0			/* Create if new */
  gDbgLvl = 0				/* Debug messages */
  gForce = 0				/* Force add */
  gAddWhere = ''			/* Position, p)refix, s)uffix */
  gVerbose = 0				/* Verbose messages */
  gEnvVar = ''				/* PATH variable to update */
  gDirectory = ''			/* Directory to add */
  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'c' then
    gCreateIfNew = 1
  when curSw == 'd' then
    gDbgLvl = gDbgLvl + 1
  when curSw == 'f' then
    gForce = 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'p' then
    gAddWhere = curSw
  when curSw == 's' then
    gAddWhere = curSw
  when curSw == 'v' then
    gVerbose = 1
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg

  /* Arg is environment variable name or pathspec */
  s = NormalizeDirName(curArg)
  /* If could be either assume environment variable if first arg and variable exists */
  if s \== '' & gEnvVar == '' then do
    if verify(curArg, gAllowedEnvVarChars, 'N') = 0 then
      if value(curArg,, gEnv) \= '' then
	s = ''				/* Treat as envvar */
  end
  if s = '' then do
    /* Not a directory, check if could be environment variable */
    if pos('\', curArg) >0 | pos(':', curArg) > 0 then
      call ScanArgsUsage curArg 'looks like a directory, but does not exist'
    if verify(curArg, gAllowedEnvVarChars, 'N') \= 0 then
      call ScanArgsUsage curArg 'cannot be an environment variable'
    if gEnvVar \== '' then
      call ScanArgsUsage 'only one environment variable allowed'
    u = translate(curArg)
    if u == 'LIBPATH' then
      curArg = 'BEGINLIBPATH'
    else if u \== 'BEGINLIBPATH' & u \== 'ENDLIBPATH' then do
      s = value(curArg,, gEnv)
      if s == '' & \ gCreateIfNew then
	call ScanArgsUsage curArg 'not in environment'
    end
    gEnvVar = curArg
    if gDbgLvl > 0 then
      say '* var' gEnvVar
  end
  else do
    if gDirectory \= '' then
      call ScanArgsUsage 'only one directory allowed'
    gDirectory = ChopDirSlash(s)
    if gDbgLvl > 0 then
      say '* directory' gDirectory
  end

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)
  if gDirectory = '' then
    call ScanArgsUsage 'directory required'
  if gEnvVar = '' then
    gEnvVar = 'PATH'
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Add directory to PATH-like variable or LIBPATH'
  say
  say 'Usage:' gCmdName '[-c] [-d] [-f] [-h] [-p] [-s] [-v] [-V] [env-var] directory'
  say
  say ' -c         Create if new'
  say ' -d         Enable debug output'
  say ' -f         Force add, default adds new only'
  say ' -h         Display this message'
  say ' -p         Add as prefix (default)'
  say ' -s         Add as suffix'
  say ' -v         Display verbose messages'
  say ' -V         Display version'
  say
  say ' env-var    Environment variable (default is PATH)'
  say ' directory  Directory name'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-c] [-d] [-f] [-h] [-p] [-s] [-v] [-V] [env-var] directory'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== AddDirSlash(directory) Append trailing \ to directory name unless just drive ===*/

AddDirSlash: procedure
  parse arg dir
  ch = right(dir, 1)
  if dir \== '' & ch \== '\' & ch \== ':' then
    dir = dir || '\'
  return dir

/* end AddDirSlash */

/*=== ChopDirSlash(directory) Chop trailing \ from directory name unless root ===*/

ChopDirSlash: procedure
  parse arg dir
  if right(dir, 1) == '\' & right(dir, 2) \== ':\' & dir \== '\' then
    dir = substr(dir, 1, length(dir) - 1)
  return dir

/* end ChopDirSlash */

/*=== NormalizeDirName(dirName) return normalized directory name with trailing \ or empty string if not directory ===*/

NormalizeDirName: procedure expose (Globals)

  /* requires AddDirSlash ChopDirSlash
   * wildcards OK in last component if wildcard resolves to single directory
   * . OK and resolves to current directory
   */
  parse arg s
  dir = ''

  if s \== '' then do
    s = ChopDirSlash(s)
    needNul = right(s, 1) == ':'
    if s == '.' then
      s = directory()
    else if s == '..' | left(s, 3) == '..\' then do
      o = directory()			/* Save */
      s = directory(s)
      call directory o			/* Restore */
      needNul = right(s, 1) == '\'
    end
  end

  if s \== '' then do
    if needNul then do
      s = s'NUL'
      call SysFileTree s, 's'
    end
    else
      call SysFileTree s, 's', 'D'
    if RESULT \= 0 then
      call Die 'SysFileTree' s 'failed'
    if s.0 = 1 then do
      parse var s.1 . . . . s
      s = strip(s)
      if needNul then
	s = left(s, lastpos('\', s))	/* drop NUL */
      dir = AddDirSlash(s)
    end
  end

  return dir

/* end NormalizeDirName */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* The end */
