#ifndef __DUMPER_H__
#define __DUMPER_H__

#include <os2.h>

#ifdef __cplusplus
extern "C" {
#endif

/********************************************************************
* _SETXCPTHANDLER will register an exception handler to write a dump
* whenever the thread, which has issued this call, traps.
*
* parameter "flag" can take the value EXIT_THREAD to indicate that only the
*           thread where the exception occurred should be terminated.
*           If this bit is not set, the process will be terminated.
*           A new file can be written each time by specifying
*           DUMPER_NEWFILE. In this case up to 10 dump files will be
*           written to the dump directory.
*           The directory for the dumps can be set using the environment
*           variable SEEDUMPDUMPDIR. If not set, the current directory is
*           used.
*           If a trap occurs, some information is displayed using
*           printf. This is set using the DUMPER_VERBOSE parameter.
*           The DUMPER_EXITLIST displays some information as the process
*           terminates.
*
* parameter "DoTheDumpEnv" points to a string which can be used to specify
*           an environment variable. When a trap occurs, this variable is
*           checked.  If it is not set, the exception is passed to OS/2.  If
*           it is set, the dump file is written.  For example, use
*           DoTheDumpEnv = "WRITE_DUMP" to write a dump file only if the
*           environment variable "WRITE_DUMP" is set.  The empty string ""
*           is interpreted to always take a dump.
*
* parameter "EndThreadHandler" can be used with the DUMPER_EXITTHREAD
*           parameter. This should contain the address of the function
*           to call to terminate the thread. Default is DosExit( EXIT_THREAD ).
*
* parameter "regrec" must point to the struct SEEDUMPREGREC
*           which must be addressable when an exception occurs.
*           The best place to declare it, is at the very beginning of main
*           or the entry routine of a thread.
********************************************************************/
#define DUMPER_VERSION     0x00010001
#define DUMPER_EXITTHREAD  0x00000001
#define DUMPER_EXITLIST    0x00000002
#define DUMPER_VERBOSE     0x00000004
#define DUMPER_NEWFILE     0x00000008

#define DUMPER_DEFAULT     (DUMPER_EXITTHREAD | DUMPER_EXITLIST \
                             | DUMPER_VERBOSE | DUMPER_NEWFILE)

typedef struct
{
  PVOID volatile prev_structure;
  PVOID volatile ExceptionHandler;
  ULONG          flag;
  PCHAR          DoTheDumpEnv;
  ULONG          EndThreadHandler;
} SEEDUMPREGREC, * PSEEDUMPREGREC;

APIRET APIENTRY _SETXCPTHANDLER( ULONG version, PSEEDUMPREGREC regrec );
#ifdef __cplusplus
}
#endif

/********************************************************************
*  the following define must be the first statement right after all
*  data declarations in main !!
*  alternatively without the macro:
*  one can declare the ExceptReg somewhere and
*  call _SETXCPTHANDLER() directly
********************************************************************/

/********************************************************************
*  the macro given below
*  will register an exception handler, that will not terminate
*  the whole process but simply the failing thread
*  in case the failing thread is thread 1, the whole process will be
*  terminated anyway.
********************************************************************/
#define SetXCPTHandler()                                            \
  SEEDUMPREGREC ExceptReg;                                          \
  {                                                                 \
     ExceptReg.prev_structure   = (PVOID)0;                         \
     ExceptReg.ExceptionHandler = (PVOID)0;                         \
     ExceptReg.flag             = DUMPER_DEFAULT;                   \
     ExceptReg.DoTheDumpEnv     = (PCHAR)0;                         \
     ExceptReg.EndThreadHandler = 0;                                \
     _SETXCPTHANDLER( DUMPER_VERSION, &ExceptReg );                 \
  }

/********************************************************************
*  the macro given below
*  will register an exception handler, that will end the failing
*  thread with a given routine specified in et
********************************************************************/
#define SetEndThreadXCPTHandler( et )                               \
  SEEDUMPREGREC ExceptReg;                                          \
  {                                                                 \
     ExceptReg.prev_structure   = (PVOID)0;                         \
     ExceptReg.ExceptionHandler = (PVOID)0;                         \
     ExceptReg.flag             = DUMPER_DEFAULT;                   \
     ExceptReg.DoTheDumpEnv     = (PCHAR)0;                         \
     ExceptReg.EndThreadHandler = (ULONG)et;                        \
     _SETXCPTHANDLER( DUMPER_VERSION, &ExceptReg );                 \
  }

/********************************************************************
*  if one wants to register the exception handler on demand
*  it is recommended to use an environment variable e.g.:
*  EXCEPTIONREGISTRATIONRECORD ExceptReg;
*  ...
*  if (getenv( "REGISTER_DUMPER") )
*  {
*    _SETXCPTHANDLER( DUMPER_VERSION, &ExceptReg );
*  }
********************************************************************/

#endif
