/******************************************************************************
  HPFSRem.c
  Allows some removable media drives to retain removability with HPFS formatted
  disks. Tested on SyQuest's EZ135 Drive. Should work on all SyQuest drives.
  May work on other drives which can be formatted with HPFS.

  By: Jeffrey Jackowski
      jackowskij@email.uah.edu
      http://www.cs.uah.edu/cs/students/jjackows/hpfsrem
       (or just search for HPFSRem on Yahoo)

  Version 1     relased July 1996
  Version 1.1   relased August 1996
    Improves handling for drives without lock/unlock capability
    Better support for removable drives with drivers that neglect removability
    Error codes from OS/2 API now reported
    New disk status feature
    Revised documentation
  Version 1.11  relased October 1996
    Includes source code
    Now under the GNU General Public License, which follows:

 ************ License ************ 

    HPFSRem -- take out an HPFS disk without a system crash
    Copyright (C) 1996  Jeffrey Jackowski

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 ************ Note from the author ************ 

    I didn't bother to clean my code up -- its unchanged except for this
    comment and a couple other things. So, the rest of this program might
    not have the best comments and is doubtless not the best example of
    how to write code, but its not the worst, either.

    Anyway, it works on some drives and not others. Maybe someone out
    there can make it work better.

******************************************************************************/

#define VERSION 1.11

#define INCL_DOSDEVICES   /* Device values */
#define INCL_DOSERRORS    /* Error values */
#define INCL_DOSDEVIOCTL  /* IO control stuff */
#include <os2.h>
#include <stdio.h>

/* Drive status flags */
#define NoLock       0
#define NoStat       3
#define DrvLocked    1
#define DrvUnlocked  2
#define DiskPresent  4

/* These defines have been made by IBM, but are not always found in other
   compilers, like the GNU/EMX C compiler */
#ifndef DSK_UNLOCKEJECTMEDIA
#define DSK_UNLOCKEJECTMEDIA 0x040
#endif
#ifndef DSK_GETLOCKSTATUS
#define DSK_GETLOCKSTATUS 0x066
#endif

void DispHelp() {  /* show a little help screen */
  printf("\nIncorrect syntax\n"
           "Please type \"HPFSRem <d> <f>\"\n"
           "where:\n"
           "\t<d> is the removable drive\n"
           "\t<f> is one of the following functions:\n"
           "\t\tR - remove disk\n"
           "\t\tI - insert disk\n"
           "\t\tRI - remove & insert disk\n"
           "\t\tS - drive info\n");
}

void DriveInfo(UCHAR driveNumber, UCHAR isRemov) {
  HFILE RemDisk =-1L;
  UCHAR zero = 0, data, mediaCon[2];
  ULONG len = 1;
  APIRET ret;

  printf("\nDrive information as reported by OS/2:\n");
  mediaCon[0] = 0;    /* reserved */
  mediaCon[1] = driveNumber; 
  if (isRemov) printf("\tRemovable\n");
  else printf("\tNon-removable  (If you can remove the disk, it will still work)\n");
  ret = DosDevIOCtl(RemDisk,IOCTL_DISK,DSK_GETLOCKSTATUS,&mediaCon,2,&len,&data,1,&len);
  if (ret == NO_ERROR) {
    if (data & NoStat)           printf("\tWill not report lock status\n");
    else if (data & DrvLocked)   printf("\tDrive locked\n");
    else if (data & DrvUnlocked) printf("\tDrive unlocked\n");
    else if (data & NoLock)      printf("\tCannot be locked\n");
    if (data & DiskPresent)      printf("\tDisk in drive\n");
  }
  if (!isRemov | ((ret == NO_ERROR) & !(data & NoLock)))
    printf("\nDrive supports HPFS removability if it can be formatted with HPFS");
  else printf("\nDrive *may* not support HPFS removability");
  printf("\n  Consult documentation for details\n");
}

int Insert(HFILE *drive, UCHAR driveNumber, UCHAR isRemov) { /* handles the insertion of disks */
  UCHAR zero = 0, data, mediaCon[2];
  ULONG len = 1;
  APIRET ret;

  mediaCon[0] = 0;    /* reserved */
  mediaCon[1] = driveNumber;
  printf("Place a disk in the drive and press enter to continue.");
  getchar();       /* wait for key press */
  ret = DosDevIOCtl(*drive,IOCTL_DISK,DSK_GETLOCKSTATUS,&mediaCon,2,&len,&data,1,&len);
  if ((ret != NO_ERROR) & ((data & DrvUnlocked) | (data & NoStat))) {  /* lock the disk */
    mediaCon[0] = 1;    /* lock command */
    ret = DosDevIOCtl(*drive,IOCTL_DISK,DSK_UNLOCKEJECTMEDIA,&mediaCon,2,&len,&data,1,&len);
    if (ret != NO_ERROR) return ret;  /* lock disk in drive to prevent removal */
  }
  /* figure out media type and attach drive to correct file system */
  ret = DosDevIOCtl(*drive,IOCTL_DISK,DSK_REDETERMINEMEDIA,&zero,1,&len,&data,1,&len);
  if (ret != NO_ERROR) return ret;
  return NO_ERROR;
}

int Remove(HFILE *drive, UCHAR driveNumber, UCHAR isRemov) { /* allows the safe removal of disks */
/* NOTE: some drives reported as non-removable can be locked or unlocked
   (ex: Zip), as a result, isRemov is currently not use in this function */
  UCHAR zero = 0, data, mediaCon[2];
  ULONG len = 1;
  APIRET ret;

  /* write all data waiting in buffer to the disk to prevent data loss */
  if (DosResetBuffer(*drive) != NO_ERROR) return -1; 
  /* prepare HPFS disk for removal -- properly stops file system */
  ret = DosDevIOCtl(*drive,IOCTL_DISK,DSK_REDETERMINEMEDIA,&zero,1,&len,&data,1,&len);
  if (ret != NO_ERROR) return ret;
  mediaCon[0] = 0;    /* reserved & unlock command */
  mediaCon[1] = driveNumber;
  ret = DosDevIOCtl(*drive,IOCTL_DISK,DSK_GETLOCKSTATUS,&mediaCon,2,&len,&data,1,&len);
  if ((ret == NO_ERROR) && ((data & NoStat) | (data & DrvLocked))) {
    /* unlock drive to allow the removal of the disk */
    ret = DosDevIOCtl(*drive,IOCTL_DISK,DSK_UNLOCKEJECTMEDIA,&mediaCon,2,&len,&data,1,&len);
    if (ret != NO_ERROR) return ret;
  }
  printf("You may now remove the disk\n");
  return NO_ERROR;
}

int main(int numParams, char *params[]) {
  HFILE RemDisk = -1L;
  ULONG action;
  UCHAR zero = 0, data, drive[3], driveNumber, opt[2], isRemov;
  ULONG len = 1, x;
  int ret;

  printf("HPFSRem -- Removability for HPFS!     Version %.2f\n"
         "  By: Jeffrey Jackowski   Copyright 1996\n"
         "HPFSRem comes with ABSOLUTELY NO WARRANTY\n"
         "This is free software, and you are welcome to redistribute it with all\n"
         "copyright and license notices intact\n", VERSION);
  if (numParams != 3) { /* make certain correct number of parameters recived */
    DispHelp();
    return -1;  /* give error code */
  }
  /****** translate the drive letter the user gave into something usable */
  drive[0] = params[1][0];
  if (drive[0] > 'Z') driveNumber = drive[0] - 'a'; /* fix case of letter */
  else driveNumber = drive[0] - 'A';                /* to find drive's #  */
  if (driveNumber > 26) { /* check for valid input */
    DispHelp();
    return -1;
  }
  drive[1] = ':';     /* complete string to open drive */
  drive[2] = 0;
  opt[0] = 0;         /* set command info for DosDevIOCtl call below */
  opt[1] = driveNumber;
  /* find out if the disk is removable according to the drivers */
  DosDevIOCtl(RemDisk,IOCTL_DISK,DSK_BLOCKREMOVABLE,&opt,2,&len,&data,1,&len);
  isRemov = !data;
  opt[0] = opt[1] = 0;    /* init option flags */
  for (x = 0; x < 2; x++) {  /* figure out user's instructions (2nd param) */
    switch (params[2][x]) {
      case 'R':              /* remove disk */
      case 'r':
        opt[0] = 1;             /* set option flag 0 */
        break;
      case 'I':              /* insert disk */
      case 'i':
        opt[1] = 1;             /* set option flag 1 */
        break;
      case 'S':              /* display info about disk */
      case 's':
        DriveInfo(driveNumber,isRemov);
        return 0;
      case 0:        /* takes care of null termination if */
        break;       /*  parameter is 1 char long         */
      default:
        DispHelp();  /* invalid option - help user out */
        return -1;
    }
  }
  /****** attempt to open drive */
  if (DosOpen(drive,&RemDisk,&action,0L,0L, OPEN_ACTION_OPEN_IF_EXISTS,
    OPEN_FLAGS_DASD | OPEN_FLAGS_FAIL_ON_ERROR | OPEN_SHARE_DENYREADWRITE |
    OPEN_ACCESS_READWRITE ,0L) != NO_ERROR) {
       printf("Error: could not access drive %s\n",drive);
       return -1;
  }
  /******* attempt to gain exclusive access to the drive */
  if (DosDevIOCtl(RemDisk,IOCTL_DISK,DSK_LOCKDRIVE,&zero,1,&len,&data,1,&len) != NO_ERROR) {
    printf("Error: could not gain exclusive access to drive %s\n",drive);
    return -1;
  }
  if (opt[0]) {                             /* remove disk */
    ret = Remove(&RemDisk,driveNumber,isRemov);
    if (ret != NO_ERROR) {
      printf("Error: could not prepare disk for removal, code %i\n",ret);
      return -1;
    }
  }
  if (opt[1]) {                             /* insert disk */
    ret = Insert(&RemDisk,driveNumber,isRemov);
    if (ret != NO_ERROR) {
      printf("Error: cannot allow disk to be inserted, code %i\n",ret);
      return -1;
    }
  }
  /****** release exclusive access to the drive */
  if (DosDevIOCtl(RemDisk,IOCTL_DISK,DSK_UNLOCKDRIVE,&zero,1,&len,&data,1,&len) != NO_ERROR) {
    printf("Error: could not release access to drive %s\n",drive);
    return -1;
  }
  /****** close the drive */
  if (DosClose(RemDisk) != NO_ERROR) {
    printf("Error: could not close drive %s\n",drive);
    return -1;
  }
  printf("Task completed successfully\n");
  return 0;
}
